package hr.tis.mgw.client.samples.send;

import hr.tis.mgw.client.GatewayException;
import hr.tis.mgw.client.charging.Charging;
import hr.tis.mgw.client.charging.ChargingManager2;
import hr.tis.mgw.client.charging.ChargingManager2Impl;
import hr.tis.mgw.client.message.MessageId;
import hr.tis.mgw.client.message.MessageManager2;
import hr.tis.mgw.client.message.MessageManager2Impl;
import hr.tis.mgw.client.message.Priority;
import hr.tis.mgw.client.message.SmsMessage;

import java.math.BigDecimal;
import java.util.Properties;

public class ChargingSamples extends AbstractSamples {

	public static final String CH1_USER = "ch1.user";
	public static final String CH1_AMOUNT = "ch1.amount";
	public static final String CH1_BILLING_TEXT = "ch1.billtext";
	public static final String CH1_REFERENCE_CODE = "ch1.refcode";

	public static final String CH2_USER = "ch2.user";
	public static final String CH2_AMOUNT = "ch2.amount";
	public static final String CH2_RESERVE_BILLING_TEXT = "ch2.reserve.billtext";
	public static final String CH2_CHARGE_REFERENCE_CODE = "ch2.charge.refcode";
	public static final String CH2_CHARGE_BILLING_TEXT = "ch2.charge.billtext";
	public static final String SMS2_FROM = "sms2.from";
	public static final String SMS2_TO = "sms2.to";
	public static final String SMS2_TEXT = "sms2.text";

	public ChargingManager2 chargingManager2;
	public MessageManager2 messageManager2;

	/*
	 * Initialize ChargingManager2.
	 * This has to be done only once per application.
	 * After initialization ChargingManager2 can be used for multiple charging requests.
	 */
	public void initChargingManager2(String url, String username, String password) {
		chargingManager2 = new ChargingManager2Impl(url, username, password);
	}

	/*
	 * Initialize MessageManager2.
	 * This has to be done only once per application.
	 * After initialization MessageManager2 can be used to send multiple sms and mms messages.
	 */
	public void initMessageManager2(String url, String username, String password) {
		messageManager2 = new MessageManager2Impl(url, username, password);
	}

	/*
	 * Charge Amount.
	 */
	public void chargeAmount(Properties prop) {
		// ----------------------
		// Contact your Mobile Operator about valid amount values and what to put in billing text and reference code.
		// ----------------------

		// ----------------------
		// Initialization (getting values for charging)
		// ----------------------
		String user = prop.getProperty(CH1_USER); // user to charge
		BigDecimal amount = new BigDecimal(prop.getProperty(CH1_AMOUNT)); // amount to charge
		String billingText = prop.getProperty(CH1_BILLING_TEXT); // billing text
		String referenceCode = prop.getProperty(CH1_REFERENCE_CODE); // reference code

		System.out.println();
		System.out.println("----------------------------------------------------");
		System.out.println("Charging Amount ->");
		System.out.println("  User: " + user);
		System.out.println("  Amount: " + amount);
		System.out.println("  Billing Text: " + billingText);
		System.out.println("  Reference Code: " + referenceCode);
		System.out.println();

		// ----------------------
		// Here is the actual code required to charge amount
		// ----------------------
		try {
			chargingManager2.chargeAmount(user, amount, billingText, referenceCode);
			System.out.println("Amount Charged.");
		}
		catch (GatewayException e) {
			System.err.println("Exception while charging amount ->");
			e.printStackTrace(System.err);
		}
	}

	/*
	 * Send SMS message and Charge Amount.
	 */
	public void sendSmsMessageAndChargeReservationAmount(Properties prop) {
		// ----------------------
		// Contact your Mobile Operator about valid amount values and what to put in billing text and reference code.
		// ----------------------

		// ----------------------
		// Initialization (getting values for charging)
		// ----------------------
		String user = prop.getProperty(CH2_USER); // user to charge
		BigDecimal amount = new BigDecimal(prop.getProperty(CH2_AMOUNT)); // amount to charge
		String billingText = prop.getProperty(CH2_RESERVE_BILLING_TEXT); // billing text
		String chargeReferenceCode = prop.getProperty(CH2_CHARGE_REFERENCE_CODE); // charge reference code
		String chargeBillingText = prop.getProperty(CH2_CHARGE_BILLING_TEXT); // charge billing text

		System.out.println();
		System.out.println("----------------------------------------------------");
		System.out.println("Charging Reservation Amount ->");
		System.out.println("  User: " + user);
		System.out.println("  Amount: " + amount);
		System.out.println("  Billing Text: " + billingText);
		System.out.println("  Charge Reference Code: " + chargeReferenceCode);
		System.out.println("  Charge Billing Text: " + chargeBillingText);

		// ----------------------
		// Initialization (getting values for sending sms message)
		// ----------------------
		String text = prop.getProperty(SMS2_TEXT); // sms message text
		String orig = prop.getProperty(SMS2_FROM); // originator address
		String[] dest = arrayFromPropertyList(prop.getProperty(SMS2_TO)); // destination address set (array)
		Priority priority = null; // ignored for SMS messages (used only for MMS messages)
		Charging charging = null; // will be assigned by Reserve Amount call on ChargingManager2

		System.out.println();
		System.out.println("----------------------------------------------------");
		System.out.println("Sending SMS message ->");
		System.out.println("  From: " + orig);
		System.out.println("  To: " + stringArrayToString(dest));
		System.out.println("  Text: " + text);
		System.out.println();

		// ----------------------
		// Here is the actual code required to reserve amount, send sms message and charge reservation
		// ----------------------
		try {
			charging = chargingManager2.reserveAmount(user, amount, billingText);

			System.out.println("Reserve Amount successful ->");
			System.out.println("  " + charging);

			SmsMessage smsMessage = new SmsMessage(text); // sms message object containing sms message text
			MessageId id = messageManager2.sendMessage(smsMessage, dest, orig, priority, charging);

			System.out.println("Message successfully sent ->");
			System.out.println("  " + id);

			chargingManager2.chargeAmountReservation(charging, amount, chargeBillingText, chargeReferenceCode);

			System.out.println("Charged Reserved Amount.");
		}
		catch (GatewayException e) {
			System.err.println("Exception in reserve-send-charge operation ->");
			e.printStackTrace(System.err);
		}
	}

	/*
	 * Application START method.
	 */
	public static void main(String[] args) {
		ChargingSamples samples = new ChargingSamples();

		// read and show properties from configuration file
		Properties prop = samples.readAndShowProperties("conf/charging_sample.conf", args);

		// properties for samples
		String cm2Url = prop.getProperty(CM2_URL);
		String cm2Username = prop.getProperty(CM2_USERNAME);
		String cm2Password = prop.getProperty(CM2_PASSWORD);

		String mm2Url = prop.getProperty(MM2_URL);
		String mm2Username = prop.getProperty(MM2_USERNAME);
		String mm2Password = prop.getProperty(MM2_PASSWORD);

		// initialize ChargingManager2
		samples.initChargingManager2(cm2Url, cm2Username, cm2Password);

		// initialize MessageManager2
		samples.initMessageManager2(mm2Url, mm2Username, mm2Password);

		// ChargingManager2 support full ParlayX charging interface.
		// Many methods don't have to be supported with you Mobile Operator.
		// Contact Mobile Operator to find which methods are supported.

		// charge amount - example 1
		if (RUN_EXAMPLE.equalsIgnoreCase(prop.getProperty(EXAMPLE_1))) {
			samples.chargeAmount(prop);
		}

		// send sms message and charge amount - example 2
		if (RUN_EXAMPLE.equalsIgnoreCase(prop.getProperty(EXAMPLE_2))) {
			samples.sendSmsMessageAndChargeReservationAmount(prop);
		}
	}
}
