package hr.tis.mgw.client.samples.send;

import hr.tis.mgw.client.GatewayException;
import hr.tis.mgw.client.charging.Charging;
import hr.tis.mgw.client.message.DeliveryStatusInfo;
import hr.tis.mgw.client.message.MessageId;
import hr.tis.mgw.client.message.MessageManager2;
import hr.tis.mgw.client.message.MessageManager2Impl;
import hr.tis.mgw.client.message.Priority;
import hr.tis.mgw.client.message.ReceivedSms;
import hr.tis.mgw.client.message.SmsMessage;

import java.util.Properties;

public class SmsSamples extends AbstractSamples {

	public static final String SMS1_FROM = "sms1.from";
	public static final String SMS1_TO = "sms1.to";
	public static final String SMS1_TEXT = "sms1.text";

	public static final String SMS2_FROM = "sms2.from";
	public static final String SMS2_TO = "sms2.to";
	public static final String SMS2_TEXT = "sms2.text";
	public static final String SMS2_SLEEP = "sms2.sleep";

	public static final String SMS3_REG_ID = "sms3.regid";

	public MessageManager2 messageManager2;

	/*
	 * Initialize MessageManager2.
	 * This has to be done only once per application.
	 * After initialization MessageManager2 can be used to send multiple sms and mms messages.
	 */
	public void initMessageManager2(String url, String username, String password) {
		messageManager2 = new MessageManager2Impl(url, username, password);
	}

	/*
	 * Send SMS message.
	 */
	public void sendSmsMessage(Properties prop) {
		// ----------------------
		// Initialization (getting values for sending message)
		// ----------------------
		String text = prop.getProperty(SMS1_TEXT); // sms message text
		String orig = prop.getProperty(SMS1_FROM); // originator address
		String[] dest = arrayFromPropertyList(prop.getProperty(SMS1_TO)); // destination address set (array)
		Priority priority = null; // ignored for SMS messages (used only for MMS messages)
		Charging charging = null; // charging is not specified in this example. See ChargingSamples for usage of charging in message sending.

		System.out.println();
		System.out.println("----------------------------------------------------");
		System.out.println("Sending SMS message ->");
		System.out.println("  From: " + orig);
		System.out.println("  To: " + stringArrayToString(dest));
		System.out.println("  Text: " + text);
		System.out.println();

		// ----------------------
		// Here is the actual code required to send sms message
		// ----------------------
		try {
			SmsMessage smsMessage = new SmsMessage(text); // sms message object containing sms message text
			MessageId id = messageManager2.sendMessage(smsMessage, dest, orig, priority, charging);

			System.out.println("Message successfully sent ->");
			System.out.println("  " + id);
		}
		catch (GatewayException e) {
			System.err.println("Exception while sending message ->");
			e.printStackTrace(System.err);
		}
	}

	/*
	 * Send SMS message with requesting delivery status.
	 */
	public void sendSmsMessageWithDeliveryStatus(Properties prop) {
		// ----------------------
		// Initialization (getting values for sending message)
		// ----------------------
		String text = prop.getProperty(SMS2_TEXT); // sms message text
		String orig = prop.getProperty(SMS2_FROM); // originator address
		String[] dest = arrayFromPropertyList(prop.getProperty(SMS2_TO)); // destination address set (array)
		Priority priority = null; // ignored for SMS messages (used only for MMS messages)
		Charging charging = null; // charging is not specified in this example. See ChargingSamples for usage of charging in message sending.
		long sleep = Long.parseLong(prop.getProperty(SMS2_SLEEP)); // sleep after sending sms message and requesting message delivery status

		System.out.println();
		System.out.println("----------------------------------------------------");
		System.out.println("Sending SMS message with STATUS ->");
		System.out.println("  From: " + orig);
		System.out.println("  To: " + stringArrayToString(dest));
		System.out.println("  Text: " + text);
		System.out.println("  Sleep: " + sleep);
		System.out.println();

		// ----------------------
		// Here is the actual code required to send sms message and request message delivery status
		// ----------------------
		try {
			SmsMessage smsMessage = new SmsMessage(text); // sms message object containing sms message text
			MessageId id = messageManager2.sendMessage(smsMessage, dest, orig, priority, charging);

			System.out.println("Message successfully sent ->");
			System.out.println("  " + id);

			// This is just for example.
			// In production application waiting (sleeping and frozing thread) for message delivery is bad.
			// Rather use another thread or design application differently to ask for delivery status when needed.
			System.out.println();
			System.out.println("Waiting for messages to be delivered ...");
			System.out.println();
			sleep(sleep);

			// ----------------------
			// Requesting message delivery status
			// ----------------------
			DeliveryStatusInfo[] infos = messageManager2.getMessageDeliveryStatus(id);

			System.out.println("Message delivery status ->");
			for (int i = 0; i < infos.length; i++) {
				System.out.println("  " + infos[i]);
			}
		}
		catch (GatewayException e) {
			System.err.println("Exception while sending message ->");
			e.printStackTrace(System.err);
		}
	}

	/*
	 * Retrieve SMS messages
	 */
	public void retrieveSmsMessages(Properties prop) {
		// ----------------------
		// Initialization (getting values for retrieving messages)
		// ----------------------
		String regId = prop.getProperty(SMS3_REG_ID);

		System.out.println();
		System.out.println("----------------------------------------------------");
		System.out.println("Retrieving SMS messages ->");
		System.out.println("  Registration Identifier: " + regId);
		System.out.println();

		// ----------------------
		// Here is the actual code required to retrieve sms messages
		// ----------------------
		try {
			ReceivedSms[] receivedSms = messageManager2.receiveSmsMessages(regId);

			System.out.println("Received Sms Messages ->");
			System.out.println("  Count: " + receivedSms.length);
			for (int i = 0; i < receivedSms.length; i++) {
				System.out.println("  " + receivedSms[i]);
			}
		}
		catch (GatewayException e) {
			System.err.println("Exception while retrieving messages ->");
			e.printStackTrace(System.err);
		}
	}

	/*
	 * Application START method.
	 */
	public static void main(String[] args) {
		SmsSamples samples = new SmsSamples();

		// read and show properties from configuration file
		Properties prop = samples.readAndShowProperties("conf/sms_sample.conf", args);

		// properties for samples
		String url = prop.getProperty(MM2_URL);
		String username = prop.getProperty(MM2_USERNAME);
		String password = prop.getProperty(MM2_PASSWORD);

		// initialize MessageManager2
		samples.initMessageManager2(url, username, password);

		// send sms - example 1
		if (RUN_EXAMPLE.equalsIgnoreCase(prop.getProperty(EXAMPLE_1))) {
			samples.sendSmsMessage(prop);
		}

		// send sms - example 2
		if (RUN_EXAMPLE.equalsIgnoreCase(prop.getProperty(EXAMPLE_2))) {
			samples.sendSmsMessageWithDeliveryStatus(prop);
		}

		// receive sms messages - example 3
		if (RUN_EXAMPLE.equalsIgnoreCase(prop.getProperty(EXAMPLE_3))) {
			samples.retrieveSmsMessages(prop);
		}
	}
}
