package hr.tis.mgw.client.samples.send;

import hr.tis.mgw.client.GatewayException;
import hr.tis.mgw.client.charging.Charging;
import hr.tis.mgw.client.message.MessageId;
import hr.tis.mgw.client.message.MessageManager2;
import hr.tis.mgw.client.message.MessageManager2Impl;
import hr.tis.mgw.client.message.MmsMessage;
import hr.tis.mgw.client.message.MmsPart;
import hr.tis.mgw.client.message.Priority;
import hr.tis.mgw.client.message.ReceivedMms;

import java.util.Properties;

public class MmsSamples extends AbstractSamples {

	public static final String MMS1_FROM = "mms1.from";
	public static final String MMS1_TO = "mms1.to";
	public static final String MMS1_SUBJECT = "mms1.subject";
	public static final String MMS1_PART1_TEXT = "mms1.part1.text";

	public static final String MMS2_FROM = "mms2.from";
	public static final String MMS2_TO = "mms2.to";
	public static final String MMS2_SUBJECT = "mms2.subject";
	public static final String MMS2_PART1_TEXT = "mms2.part1.text";
	public static final String MMS2_PART2_IMAGE_FILENAME = "mms2.part2.filename";
	public static final String MMS2_PART2_IMAGE_CT = "mms2.part2.ct";

	public static final String MMS3_REG_ID = "mms3.regid";

	public MessageManager2 messageManager2;

	/*
	 * Initialize MessageManager2.
	 * This has to be done only once per application.
	 * After initialization MessageManager2 can be used to send multiple sms and mms messages.
	 */
	public void initMessageManager2(String url, String username, String password) {
		messageManager2 = new MessageManager2Impl(url, username, password);
	}

	/*
	 * Send MMS message with one text attachment.
	 */
	public void sendMmsMessage(Properties prop) {
		// ----------------------
		// Initialization (getting values for sending message)
		// ----------------------
		String orig = prop.getProperty(MMS1_FROM); // originator address
		String[] dest = arrayFromPropertyList(prop.getProperty(MMS1_TO)); // destination address set (array)
		String subject = prop.getProperty(MMS1_SUBJECT); // subject of mms message
		String part1Text = prop.getProperty(MMS1_PART1_TEXT); // first part of mms message - plain text
		Priority priority = null; // priority if not set - default will be used
		Charging charging = null; // charging is not specified in this example. See ChargingSamples for usage of charging in message sending.

		System.out.println();
		System.out.println("----------------------------------------------------");
		System.out.println("Sending MMS message ->");
		System.out.println("  From: " + orig);
		System.out.println("  To: " + stringArrayToString(dest));
		System.out.println("  Subject: " + subject);
		System.out.println("  Part 1 - Text: " + part1Text);
		System.out.println();

		// ----------------------
		// Here is the actual code required to send mms message
		// ----------------------
		try {
			MmsMessage mmsMessage = new MmsMessage(subject); // mms message
			MmsPart textPart = new MmsPart(part1Text); // first and only textual part
			mmsMessage.addPart(textPart); // add part to mms message
			MessageId id = messageManager2.sendMessage(mmsMessage, dest, orig, priority, charging);

			System.out.println("Message successfully sent ->");
			System.out.println("  " + id);

			// Requesting message delivery status.
			// Since delivery of MMS message can last for minutes this is not done in this example.
			// You can see in SmsSamples how it works and how to obtain message delivery status.
			// The principle is the same for MMS messages.
		}
		catch (GatewayException e) {
			System.err.println("Exception while sending message ->");
			e.printStackTrace(System.err);
		}
	}

	/*
	 * Send MMS message with binary attachments.
	 */
	public void sendMmsMessageWithBinaryAttachments(Properties prop) {
		// ----------------------
		// Initialization (getting values for sending message)
		// ----------------------
		String orig = prop.getProperty(MMS2_FROM); // originator address
		String[] dest = arrayFromPropertyList(prop.getProperty(MMS2_TO)); // destination address set (array)
		String subject = prop.getProperty(MMS2_SUBJECT); // subject of mms message
		String part1Text = prop.getProperty(MMS2_PART1_TEXT); // first part of mms message - plain text
		String part2FileName = prop.getProperty(MMS2_PART2_IMAGE_FILENAME); // second part of mms message - image filename
		String part2ContentType = prop.getProperty(MMS2_PART2_IMAGE_CT); // second part of mms message - content type
		Priority priority = Priority.NORMAL; // normal priority
		Charging charging = null; // charging is not specified in this example. See ChargingSamples for usage of charging in message sending.

		System.out.println();
		System.out.println("----------------------------------------------------");
		System.out.println("Sending MMS message ->");
		System.out.println("  From: " + orig);
		System.out.println("  To: " + stringArrayToString(dest));
		System.out.println("  Subject: " + subject);
		System.out.println("  Part 1 Text: " + part1Text);
		System.out.println("  Part 2 Image filename: " + part2FileName);
		System.out.println("  Part 2 Image content type: " + part2ContentType);
		System.out.println();

		// ----------------------
		// Here is the actual code required to send mms message
		// ----------------------
		try {
			MmsMessage mmsMessage = new MmsMessage(subject); // mms message
			mmsMessage.addPart(new MmsPart(part1Text)); // add part 1 (plain text) to mms message
			mmsMessage.addPart(new MmsPart(fileBinaryContent(part2FileName), part2ContentType)); // add part 2 (image) to mms message
			MessageId id = messageManager2.sendMessage(mmsMessage, dest, orig, priority, charging);

			System.out.println("Message successfully sent ->");
			System.out.println("  " + id);

			// Requesting message delivery status.
			// Since delivery of MMS message can last for minutes this is not done in this example.
			// You can see in SmsSamples how it works and how to obtain message delivery status.
			// The principle is the same for MMS messages.
		}
		catch (GatewayException e) {
			System.err.println("Exception while sending message ->");
			e.printStackTrace(System.err);
		}
	}

	/*
	 * Retrieve MMS messages
	 */
	public void receiveMmsMessages(Properties prop) {
		// ----------------------
		// Initialization (getting values for retrieving messages)
		// ----------------------
		String regId = prop.getProperty(MMS3_REG_ID);

		System.out.println();
		System.out.println("----------------------------------------------------");
		System.out.println("Retrieving MMS messages ->");
		System.out.println("  Registration Identifier: " + regId);
		System.out.println();

		// ----------------------
		// Here is the actual code required to retrieve mms messages
		// ----------------------
		try {
			ReceivedMms[] receivedMms = messageManager2.receiveMmsMessages(regId, Priority.DEFAULT);

			System.out.println("Received Mms Messages ->");
			System.out.println("  Count: " + receivedMms.length);
			for (int i = 0; i < receivedMms.length; i++) {
				System.out.println("  " + receivedMms[i]);
			}
		}
		catch (GatewayException e) {
			System.err.println("Exception while retrieving messages ->");
			e.printStackTrace(System.err);
		}
	}

	/*
	 * Application START method.
	 */
	public static void main(String[] args) {
		MmsSamples samples = new MmsSamples();

		// read and show properties from configuration file
		Properties prop = samples.readAndShowProperties("conf/mms_sample.conf", args);

		// properties for samples
		String url = prop.getProperty(MM2_URL);
		String username = prop.getProperty(MM2_USERNAME);
		String password = prop.getProperty(MM2_PASSWORD);

		// initialize MessageManager2
		samples.initMessageManager2(url, username, password);

		// send mms - example 1
		if (RUN_EXAMPLE.equalsIgnoreCase(prop.getProperty(EXAMPLE_1))) {
			samples.sendMmsMessage(prop);
		}

		// send mms - example 2
		if (RUN_EXAMPLE.equalsIgnoreCase(prop.getProperty(EXAMPLE_2))) {
			samples.sendMmsMessageWithBinaryAttachments(prop);
		}

		// receive mms messages - example 3
		if (RUN_EXAMPLE.equalsIgnoreCase(prop.getProperty(EXAMPLE_3))) {
			samples.receiveMmsMessages(prop);
		}
	}
}
